<?php
/**
 * Exit if accessed directly
 */
defined( 'ABSPATH' ) || exit( 'دسترسی غیر مجاز!' );

add_filter( 'woocommerce_account_menu_items', 'wast_woo_account_menu', 10, 1 );
add_action( 'init', 'wast_woo_add_tickets_endpoint' );
add_action( 'woocommerce_account_tickets_endpoint', 'wast_woo_tickets_endpoint_content' );

/**
 * Insert the new endpoint into the My Account Menu
 *
 * @param array $items
 *
 * @return array
 */
function wast_woo_account_menu( $items ) {
	if ( ! wast_is_user_seller( get_current_user_id() ) ) {
		$logout = $items['customer-logout'];
		unset( $items['customer-logout'] );
		$items['tickets']         = 'تیکت پشتیبانی';
		$items['customer-logout'] = $logout;
	}

	return $items;
}

/**
 * Adding an endpoint creates extra rewrite rules
 */
function wast_woo_add_tickets_endpoint() {
	if ( wast_is_user_seller( get_current_user_id() ) ) {
		return false;
	}

	add_rewrite_endpoint( 'tickets', EP_PAGES );
	flush_rewrite_rules();
}

/**
 * Load ticket template content
 */
function wast_woo_tickets_endpoint_content() {
	if ( wast_is_user_seller( get_current_user_id() ) ) {
		return false;
	}

	if ( ! is_user_logged_in() ) {
		echo '<p class="wast-alert wast-alert-warning">برای ثبت تیکت بایستی وارد اکانت خود شوید!</p>';

		return false;
	}

	if ( isset( $_GET['action'] ) && ! empty( $_GET['action'] ) ) {
		if ( $_GET['action'] == 'new' ) {
			wast_send_ticket();
			include( WAST_TPL_PATH . 'panel/tpl-new-ticket.php' );
		} elseif ( $_GET['action'] == 'reply' ) {
			wast_reply_ticket();
			include( WAST_TPL_PATH . 'panel/tpl-reply-ticket.php' );
		}
	} else {
		include( WAST_TPL_PATH . 'panel/tpl-all-tickets.php' );
	}
}

/**
 * Perform send new ticket
 */
function wast_send_ticket() {
	if ( isset( $_POST['submit-new-ticket'] ) ) {
		if ( ! isset( $_POST['wast_new_ticket_field'] ) || ! wp_verify_nonce( $_POST['wast_new_ticket_field'], 'wast_new_ticket' ) ) {
			exit( 'Sorry, your nonce did not verify.' );
		} else {
			$ticket_title    = sanitize_text_field( $_POST['ticket-title'] );
			$ticket_type     = $_POST['ticket-type'];
			$product_id      = $_POST['product-id'];
			$product_title   = isset( $_POST['product-title'] ) ? $_POST['product-title'] : '';
			$ticket_priority = $_POST['ticket-priority'];
			$ticket_content  = ! wast_get_option( 'wp-editor' ) ? implode( "\n", array_map( 'sanitize_text_field', explode( "\n", $_POST['ticket-content'] ) ) ) : $_POST['ticket-content'];
			// Sanitize content from script tag (XSS attack)
			$ticket_content = preg_replace( '#<script(.*?)>(.*?)</script>#is', '', $ticket_content );

			// Check user purchased products and expiration time for woocommerce
			if ( ! wast_is_customer_bought_product( $product_id ) ) {
				WAST_Flash_Message::add_message( 'شما نمی توانید از پشتیبانی این محصول استفاده کنید!', 1 );

				return false;
			}

			$has_errors = false;

			if ( ! isset( $ticket_title ) || empty( $ticket_title ) ) {
				WAST_Flash_Message::add_message( 'لطفا عنوان تیکت را وارد کنید!', 1 );
				$has_errors = true;
			} else {
				if ( strlen( $ticket_title ) > 128 ) {
					WAST_Flash_Message::add_message( 'عنوان تیکت خیلی طولانی می‌باشد!', 1 );
					$has_errors = true;
				}
			}

			if ( strlen( $product_title ) > 512 ) {
				WAST_Flash_Message::add_message( 'عنوان محصول خیلی طولانی می‌باشد!', 1 );
				$has_errors = true;
			}

			if ( ! isset( $ticket_type ) || empty( $ticket_type ) ) {
				WAST_Flash_Message::add_message( 'لطفا بخش مورد نظر را انتخاب کنید!', 1 );
				$has_errors = true;
			}

			if ( ! isset( $ticket_content ) || empty( $ticket_content ) ) {
				WAST_Flash_Message::add_message( 'لطفا متن تیکت را وارد کنید!', 1 );
				$has_errors = true;
			} else {
				if ( strlen( strip_tags( $ticket_content ) ) > 8192 ) {
					WAST_Flash_Message::add_message( 'متن تیکت خیلی طولانی می‌باشد!', 1 );
					$has_errors = true;
				}
			}

			if ( $has_errors ) {
				return false;
			}

			$ticket_post = array(
				'post_type'    => 'ticket',
				'post_title'   => $ticket_title,
				'post_content' => $ticket_content,
				'post_status'  => 'open',
				'post_author'  => get_current_user_id(),
				'meta_input'   => array(
					'_wast_ticket_priority'   => $ticket_priority,
					'_wast_ticket_product_id' => $product_id ? $product_id : $product_title,
				),
			);

			$to = get_term_meta( $ticket_type, '_wast_recipient_email', true );
			if ( wast_get_option( 'seller-support' ) && wast_get_option( 'dokan-ticket-type' ) == $ticket_type ) {
				if ( intval( $product_id ) ) {
					$seller_id = get_post_field( 'post_author', $product_id );
					$user_info = get_userdata( $seller_id );
					$to        = $user_info->user_email;
				}
			}

			$recipient        = null;
			$recipient_mobile = null;
			if ( $to ) {
				$recipient        = get_user_by( 'email', $to );
				$recipient_mobile = get_user_meta( $recipient->ID, '_wast_mobile', true );
			}
			$term = get_term( $ticket_type, 'ticket-type' );

			if ( isset( $_FILES['ticket-file']['name'] ) && ! empty( $_FILES['ticket-file']['name'] ) ) {
				$file_url = wast_handle_upload();
				if ( $file_url ) {
					$ticket_id = wp_insert_post( $ticket_post );

					if ( $term ) {
						wp_set_object_terms( $ticket_id, array( $term->name ), 'ticket-type' );
					}
					update_post_meta( $ticket_id, '_wast_ticket_ath_file', $file_url );
					update_post_meta( $ticket_id, '_wast_ticket_last_reply_date', time() );

					WAST_Flash_Message::add_message( 'تیکت شما با موفقیت ارسال گردید.', 2 );

					// Send notification to user
					do_action( 'wast_send_email', $to, $ticket_id );
					do_action( 'wast_send_sms', $recipient_mobile, $ticket_id );
				}
			} else {
				$ticket_id = wp_insert_post( $ticket_post );
				if ( $term ) {
					wp_set_object_terms( $ticket_id, array( $term->name ), 'ticket-type' );
				}
				update_post_meta( $ticket_id, '_wast_ticket_last_reply_date', time() );

				WAST_Flash_Message::add_message( 'تیکت شما با موفقیت ارسال گردید.', 2 );

				// Send notification to user
				do_action( 'wast_send_email', $to, $ticket_id );
				do_action( 'wast_send_sms', $recipient_mobile, $ticket_id );
			}
		}
	}
}

/**
 * Perform reply new ticket
 */
function wast_reply_ticket() {
	if ( isset( $_POST['submit-reply-ticket'] ) ) {
		if ( ! isset( $_POST['wast_reply_ticket_field'] ) || ! wp_verify_nonce( $_POST['wast_reply_ticket_field'], 'wast_reply_ticket' ) ) {
			exit( 'Sorry, your nonce did not verify.' );
		} else {
			$ticket_id  = $_POST['ticket-id'];
			$product_id = $_POST['product-id'];
			if ( ! intval( $ticket_id ) ) {
				return false;
			}
			$ticket_content = ! wast_get_option( 'wp-editor' ) ? implode( "\n", array_map( 'sanitize_text_field', explode( "\n", $_POST['ticket-content'] ) ) ) : $_POST['ticket-content'];
			// Sanitize content from script tag (XSS attack)
			$ticket_content = preg_replace( '#<script(.*?)>(.*?)</script>#is', '', $ticket_content );
			$ticket_content = stripslashes( $ticket_content );
			$status         = isset( $_POST['status-closed'] ) ? 'closed' : 'open';

			// Check user purchased products and expiration time for woocommerce
			if ( ! wast_is_customer_bought_product( $product_id ) ) {
				WAST_Flash_Message::add_message( 'شما نمی توانید از پشتیبانی این محصول استفاده کنید!', 1 );

				return false;
			}

			$has_errors = false;

			if ( ! isset( $ticket_content ) || empty( $ticket_content ) ) {
				WAST_Flash_Message::add_message( 'لطفا متن تیکت را وارد کنید!', 1 );
				$has_errors = true;
			} else {
				if ( strlen( strip_tags( $ticket_content ) ) > 8192 ) {
					WAST_Flash_Message::add_message( 'متن تیکت خیلی طولانی می‌باشد!', 1 );
					$has_errors = true;
				}
			}

			if ( $has_errors ) {
				return false;
			}

			$reply = array(
				'author_id' => get_current_user_id(),
				'content'   => $ticket_content,
				'date'      => time(),
			);

			$to          = '';
			$ticket_type = wp_get_post_terms( $ticket_id, 'ticket-type' );
			if ( $ticket_type && is_array( $ticket_type ) ) {
				$to = get_term_meta( $ticket_type[0]->term_id, '_wast_recipient_email', true );
				if ( wast_get_option( 'seller-support' ) && wast_get_option( 'dokan-ticket-type' ) == $ticket_type[0]->term_id ) {
					if ( intval( $product_id ) ) {
						$seller_id = get_post_field( 'post_author', $product_id );
						$user_info = get_userdata( $seller_id );
						$to        = $user_info->user_email;
					}
				}
			}

			$recipient        = null;
			$recipient_mobile = null;
			if ( $to ) {
				$recipient        = get_user_by( 'email', $to );
				$recipient_mobile = get_user_meta( $recipient->ID, '_wast_mobile', true );
			}

			$reply_count = wast_get_reply_count( $ticket_id );

			if ( isset( $_FILES['ticket-file']['name'] ) && ! empty( $_FILES['ticket-file']['name'] ) ) {
				$file_url = wast_handle_upload();
				if ( $file_url ) {
					wp_update_post( array(
						'ID'          => $ticket_id,
						'post_status' => $status
					) );
					$reply['ath_file'] = $file_url;
					update_post_meta( $ticket_id, '_wast_ticket_reply' . ( $reply_count + 1 ), $reply );
					update_post_meta( $ticket_id, '_wast_ticket_last_reply_date', time() );
					WAST_Flash_Message::add_message( 'پاسخ شما با موفقیت ارسال گردید.', 2 );

					// Send notification to user
					do_action( 'wast_send_email', $to, $ticket_id );
					do_action( 'wast_send_sms', $recipient_mobile, $ticket_id );
				}
			} else {
				wp_update_post( array(
					'ID'          => $ticket_id,
					'post_status' => $status
				) );
				update_post_meta( $ticket_id, '_wast_ticket_reply' . ( $reply_count + 1 ), $reply );
				update_post_meta( $ticket_id, '_wast_ticket_last_reply_date', time() );
				WAST_Flash_Message::add_message( 'پاسخ شما با موفقیت ارسال گردید.', 2 );

				// Send notification to user
				do_action( 'wast_send_email', $to, $ticket_id );
				do_action( 'wast_send_sms', $recipient_mobile, $ticket_id );
			}
		}
	}
}

/**
 * Upload file in wp uploads folders
 * Check file size and MIME type
 * @return bool
 */
function wast_handle_upload() {
	// Change upload directory
	add_filter( 'upload_dir', 'wast_alter_the_upload_dir' );

	if ( ! function_exists( 'wp_handle_upload' ) ) {
		require_once( ABSPATH . 'wp-admin/includes/file.php' );
	}

	$uploaded_file = $_FILES['ticket-file'];

	$max_size = wast_get_option( 'file-max-size' );
	if ( $max_size ) {
		if ( ( $uploaded_file['size'] / 1024 ) * 0.001 > $max_size ) {
			WAST_Flash_Message::add_message( 'حداکثر حجم مجاز فایل <strong>' . $max_size . '</strong> مگابایت میباشد!', 1 );

			return false;
		}
	}

	$upload_overrides = array(
		'test_form' => false,
	);

	if ( wast_get_mimes() ) {
		$upload_overrides['mimes'] = wast_get_mimes();
	}

	$move_file = wp_handle_upload( $uploaded_file, $upload_overrides );

	if ( $move_file && ! isset( $move_file['error'] ) ) {
		return $move_file['url'];
	} else {
		WAST_Flash_Message::add_message( $move_file['error'], 1 );

		return false;
	}
}